<?php
declare(strict_types=1);

require_once __DIR__ . '/db.php';

// Settings are stored in app_setting (key -> json value).
// This loader keeps a per-request cache.

function settings_all(): array {
  static $cache = null;
  if ($cache !== null) return $cache;
  $rows = db()->query('SELECT key, value_json FROM app_setting')->fetchAll();
  $out = [];
  foreach ($rows as $r) {
    $out[$r['key']] = json_decode($r['value_json'], true);
  }
  $cache = $out;
  return $out;
}

function setting(string $key, $default = null) {
  $all = settings_all();
  return array_key_exists($key, $all) ? $all[$key] : $default;
}

function public_settings(): array {
  $brand = setting('branding', [
    'appName' => 'EinsatzBoard',
    'primaryColor' => '#d6002a',
    'accentColor' => '#111827',
    'logoPath' => null,
    'faviconPath' => null,
  ]);
  $privacy = setting('privacy', [
    'pseudonymizeOtherOrgs' => true,
    'showNamesToOtherOrgs' => false,
    'retentionDays' => 180,
  ]);
  $ops = setting('ops', [
    'transferModeDefault' => 'offer', // offer|fast
    'fastModeGraceSeconds' => 120,
    'heartbeatWarnMinutes' => 30,
    'heartbeatAlarmMinutes' => 45,
  ]);

  // Team settings (kept configurable to avoid hardcoding roles like HF/NAVI)
  $team = setting('team', [
    'roles' => [
      ['key'=>'lead','label'=>'Leitung','sort'=>10],
      ['key'=>'nav','label'=>'Navigation','sort'=>20],
      ['key'=>'radio','label'=>'Funk','sort'=>30],
      ['key'=>'member','label'=>'Mitglied','sort'=>90],
    ],
    'enableLeadFlag' => true,
  ]);

  return ['branding'=>$brand, 'privacy'=>$privacy, 'ops'=>$ops, 'team'=>$team];
}

function settings_update(string $key, array $value): void {
  $stmt = db()->prepare('INSERT INTO app_setting(key, value_json, updated_at) VALUES(:k, :v::jsonb, now())
    ON CONFLICT (key) DO UPDATE SET value_json = EXCLUDED.value_json, updated_at = now()');
  $stmt->execute([':k'=>$key, ':v'=>json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)]);
}
